%% E3_Fired_Cartridge_Case_Comparison_System 
%
% version 2022-06-06a
%
% Available from http://forensic-data-science.net/firearms/
%
% Implements algorithms decribed in:
%   Basu N., Bolton-King R.S., Morrison G.S. (2022). Forensic comparison of fired cartridge cases: Feature-extraction methods for feature-based calculation of likelihood ratios. Forensic Science International: Synergy, 5, article 100272.
%   https://doi.org/10.1016/j.fsisyn.2022.100272
%
% Code developed and mainted by Dr. Nabanita Basu
%
% Version of Matlab used to develop the code: 9.10.0.1602886 (R2021a)
%
% Operating System used: Windows 10
%
% Toolboxes required:
%   Computer Vision Toolbox
%   Parallel Computing Toolbox
%   Statistics and Machine Learning Toolbox
%   Image Processing Toolbox
%
% Third party software required to open x3p files:
%   7z.exe
%   available from https://www.7-zip.org/
%   set path to location of 7z.exe on line 9 of 'readX3P_parallel.m'
%
% Data required:
%   E3 Database of Fired Cartridge Cases (release 1)
%   available via http://forensic-data-science.net/firearms/
%
% Development of the E3 Fired Cartridge Case Comparison System was supported by Research England’s Expanding Excellence in England Fund as part of funding for the Aston Institute for Forensic Linguistics 2019–2023.


clc; clear all; close all;
addpath('./functions_parallel');
addpath('./moments_parallel');

rng(0);
%registration flag
register_ROI=true;



%% List of abbreviations to be followed in the program

% wwf : segment consisting of firing pin impression, breech face region
% with flowback

% wof : segment consisting of firing pin impression, breech face region
% without flowback

% bfr : segmented breech face region only

% fpi : segmented firing pin impression only

% cct : concatenation of features extracted separately from the firing pin
%       impression and the breech face region

% scf : concatenation of scores obtained separately from the firing pin
%       impression and the breech face region

% cem : central moments

% cim : circle-moment invariants

% lmi : legendre moments

% cmi : central-moment invariants

% fma : fourier magnitude only features

% fmp : fourier magnitude and phase features

% zma : zernike magnitude only features

% zmp : zernike magnitude and phase feaures

% suffix R : used to suggest whether the segmented region has passed
%            through the registration block.
%            If 'registration flag' is set to 1 then the segment 
%            is registered to a pre-selected scan.
%            Else no registration is performed - the scan is registered
%            using the transfrmation matrix (tform) =[1 0 0 0;0 1 0 0;0 0 1
%            0;0 0 0 1]- that leads to no change in translation and
%            rotation
%
% pb : planar bias
%
% cs : circular symmetry
%            
% <feature abbreviation>_1v1 - Cllr values for cartridge case features compared using
% Model 1 v 1 eg. cem_1v1, cim_1v1 etc.

% <feature abbreviation>_1v3 - Cllr values for cartridge case features compared using
% Model 1 v 3 eg. cem_1v3, cim_1v3 etc.

% <feature abbreviation>_1v9 - Cllr values for cartridge case features compared using
% Model 1 v 9 eg. cem_1v9, cim_1v9 etc.




%segmentation flags
wwf= true;
fpi = true;
bfr=true;
wof=true;

% features
cem=true;
cim=true;
cmi=true;
lmi=true;
fma=true;
fmp=true;
zma=true;
zmp=true;



% fusions
cct=true;
scf=true;

%comparison variations
do_1v1=true;
do_1v3=true;
do_1v9=true;

% Print values to file
print2File=true;

% do randmizations
randomization=true;

flags = {'register_ROI', 'wwf', ...
    'fpi', 'bfr',...
    'wof', 'cem',...
    'cim','cmi',...
    'lmi','fma','fmp'...
    'zma','zmp','cct',...
    'scf','do_1v1','do_1v3'...
    'do_1v9','print2File','randomization'};% Creating the list of variables

save('flags', flags{:});


% location from which the x3p files are to be read
data_directory='F:\Dataset\Firearms\Firearms_0001_0203\';


s = dir(fullfile(data_directory,'*.x3p'));

% number of data files
data_files=size(s,1);



%% Step 0: select image based on which you wnt to register other images

ex=exist('pc_wwf_fixed.mat','file');

%if ex==0
% The fixed image
[y1 x1 z1]=read_in_x3p_file(data_directory,s(24).name); % Select 1st image as the fixed point for registration
[x_down1, y_down1, z_down1] = im_resample(x1,y1,z1); %resample image
[mask_wwf_fixed, cmask_wwf_fixed, boundingBox_wwf_fixed,wwf_nan_fixed,wwf_fill_rough,wwf_fill_fixed] = segment_ROI_remove_headstamp(z_down1);% segment the ROI by removal of the headstamp region

%crop that region
[xf_crop, yf_crop,~,wwf_fixed_crop,~] = crop_3D_mesh(wwf_fill_rough,x_down1,y_down1,wwf_nan_fixed);

%perform planar illumination correction
% should be done on wwf_fixed_crop
[BB, pb_corrected_wwf_fixed]=ransac_plane_fit(xf_crop,yf_crop,wwf_fixed_crop);

%perform circular symmetry correction
cs_corrected_wwf_fixed = remove_circular_symmetry(pb_corrected_wwf_fixed);

% Estimating the point cloud
pc_wwf_fixed = convert_to_point_cloud(xf_crop,yf_crop,cs_corrected_wwf_fixed);
save('pc_wwf_fixed.mat','pc_wwf_fixed');

if  fma|fmp
    %generate binary mask for ROI region
    cs_corrected_wwf_fixed_mask=generate_binary_mask(cs_corrected_wwf_fixed);
    %get scale for largest fitted circle
    max_radius=get_circle_radius(cs_corrected_wwf_fixed_mask);
end



%% for all images in the dataset
labs=zeros(data_files,1);  % Labels = Gun number from which the cartridge case has been fired
work_folder='E:\Cartridge Case Project\';


parfor (k =1:data_files,29)
 
    close all;
    
    % read in an image from database
    [~ ,fname,~]=fileparts(s(k).name);
    cd(work_folder);
    A=exist(fname,'dir');
    %if A==0
       mkdir(fname);
        N=" ";
        N=cat(2,work_folder,fname,'\');
        cd(N)
        
        %% Step 1 : To generate labels from cartridge case sample name
        lab = generate_labels(fname);
        
        %% Step 2: To read in a x3p file
        [y x z]=read_in_x3p_file(data_directory,s(k).name);
        
        
        % Step 2.1 : resample image
        [x_down y_down z_down] = im_resample(x,y,z);
        details=struct('lab',lab,'x_down',x_down,'y_down',y_down,'z_down',z_down);
        saveMat('details',details);  % Saving Variables
        
        %% Step 3 : First segmentation - segment out ROI - FPI+ Flowback +Breech Face
        [mask_wwf, cmask_wwf, boundingBox_wwf,wwf_nan,wwf_rough, wwf_fill] = segment_ROI_remove_headstamp(z_down);
        
        
        %% Step 4 : Crop Image and perform Illumination correction
        
        % Step 4.1 : First crop the image tightly
        [x_crop, y_crop,~,wwf_crop, ~] = crop_3D_mesh(wwf_rough,x_down,y_down,wwf_nan);
        
        % Step 4.2 : Planar bias correction
        [B, pb_corrected_wwf]=ransac_plane_fit(x_crop,y_crop,wwf_crop);
        
        % Step 4.3 : Perform circular symmetry correction
        cs_corrected_wwf = remove_circular_symmetry(pb_corrected_wwf);
        
        %% Step 5 : Perform Registration using ICP(rigid)
        cs_corrected_wwfR=[];
        x_downR=[];
        y_downR=[];
        
        if register_ROI
            %
            pc_wwf =convert_to_point_cloud(x_crop,y_crop,cs_corrected_wwf);
            
            [ro,co]=size(cs_corrected_wwf);
            
            [tform, pc_registered, rmse] = pcregistericp(pc_wwf,pc_wwf_fixed, 'Extrapolate', true);
            
            xyz_wwf=pc_registered.Location;
            
            x_down_a=reshape(xyz_wwf(:,1),[ro co]);
            
            y_down_a=reshape(xyz_wwf(:,2),[ro co]);
            
            x_downR=nanmean(x_down_a,2);
            
            y_downR=nanmean(y_down_a,1);
            
            cs_corrected_wwfR = reshape(xyz_wwf(:,3),[ro co]);
            
        else
            
            rot_tran=[1 0 0 0;0 1 0 0; 0 0 1 0; 0 0 0 1];
            
            tform=rigid3d(rot_tran);
            
            x_downR=x_crop;
            
            y_downR=y_crop;
            
            cs_corrected_wwfR = cs_corrected_wwf;
        end
        
        % From hereon the ROI consisting of FPI+Flowback+BreechFace will be represented by
        % cs_corrected_wwfR...
        
        %% Step 6 : Segment Firing pin impression
        
        % Step 6.1 : Segmentation of FPI from registered,non -illumination
        % corrected image
        
        % convert apIm ,apCIm and non-illumination corrected U32 using tform
        pc_mask_wwf =convert_to_point_cloud(x_crop,y_crop,mask_wwf);
        pc_cmask_wwf=convert_to_point_cloud(x_crop,y_crop,cmask_wwf);
        pc_wwf_crop=convert_to_point_cloud(x_crop,y_crop,wwf_crop);
        
        % cropped ROI from registered BF_rough
        mask_wwfR= rotate_using_tform(pc_mask_wwf,mask_wwf,tform);
        
        
        
        % cropped ROI from registered BF_fill
         cmask_wwfR=rotate_using_tform(pc_cmask_wwf,mask_wwf,tform);
        
        % register non illumination corrected ROI consisting of FPI+flowback+BF
        pc_wwf_cropR= pctransform(pc_wwf_crop,tform);
        wwf_cropR= reshape(pc_wwf_cropR.Location(:,3),[size(wwf_crop,1) size(wwf_crop,2)]);
        
        % FF1 - FPI mask from pixel based
        % FF - FPI mask from contour based
        [gray_wwf,fpi_pixel,fpi_contour,edge_wwf] = segment_fpi(wwf_cropR,mask_wwfR);
        
        % Step 6.2 : Refining FPI using illumination corrected cs_corrected_wwfR
        cs_corrected_wwfR_n=double(logical(cmask_wwfR)).*cs_corrected_wwfR;
        %
        cs_corrected_wwfR_n(cs_corrected_wwfR_n==0)=NaN;
        %
        cs_corrected_wwfR_gray=convertMat2gray(cs_corrected_wwfR_n);
        cs_corrected_wwfR_gray=uint8(255.*cs_corrected_wwfR_gray); % Grayscale representation of overall illumination corrected image w FPI
        
        mask_fpi=refined_fpi(fpi_contour,fpi_pixel,cs_corrected_wwfR_gray,edge_wwf);
        
        % Step 6.3 : Extracting only the FPI region
        fpiR=wwf_cropR.*double(mask_fpi);
        fpiR(fpiR==0)=NaN;
        [x_fpiR,y_fpiR,~, fpiR_crop, boundingBox_fpiR] = crop_3D_mesh(mask_fpi,x_downR,y_downR,fpiR);
        % binary mask of a grayscale image
        binary_fpi = generate_binary_mask(fpiR_crop);
        
        %% Step 7 : Segment Flowback region - Illumination based Flowback estimation using refined FPI
        
        if bfr | fpi | wof|cct| scf
            mask_bfr_flowback=double(logical(cmask_wwfR))-mask_fpi;
            bfr_flowbackR=mask_bfr_flowback.*cs_corrected_wwfR;
            bfr_flowbackR(bfr_flowbackR==0)=NaN;
            gray_bfr_flowbackR=convertMat2gray(bfr_flowbackR);
            gray_bfr_flowbackR=uint8(255.*gray_bfr_flowbackR); % Grayscale representation of overall illumination corrected image w/o FPI
            flowbackR = segment_flowback(gray_bfr_flowbackR,mask_fpi,edge_wwf);
            
            
            %% Step 8 : Segment Breech Face region
            %
            mask_bfr=double(logical(cmask_wwfR))-flowbackR;
            bfrR = mask_bfr.*wwf_cropR; % Breech Face region from original;
            bfrR(bfrR==0)=nan; % Final segmented Breech face region
            %
            %% Step 9 : Illumination correction of BreechFace region
            
            % Step 9.1 : % Planar bias correction of Breech face region
            [bfr_plane_coeff,pb_corrected_bfrR]=ransac_plane_fit(x_downR,y_downR,bfrR);
            
            % Step 9.2 : Circular symmetry correction of BF region
            cs_corrected_bfrR = remove_circular_symmetry(pb_corrected_bfrR);
            
        end
        
        %% Step 10 : Illumination correction of Firing pin
        %
        if fpi| wof | cct|scf
            % Step 10.1 : % Planar bias correction of firing pin impression based on plane
            % fitted to breech face region
            pb_corrected_fpiR = ransac_plane_fit_fpi(bfr_plane_coeff,x_fpiR,y_fpiR,fpiR_crop);
            
            % Step 10.2 : Circular symmetry correction of firing pin impression
            cs_corrected_fpiR = remove_circular_symmetry(pb_corrected_fpiR);
            
        end
        
           %% Step 11 : Applying Gaussian Bandpass filter for noise correction of wwf - ROI consisting of firing pin impression  and breech face region with flowback
        if wwf
            [gray_cs_corrected_wwfR,max_wwfR,min_wwfR] = convertMat2gray(cs_corrected_wwfR);
            gray_wwfR= 255.*gray_cs_corrected_wwfR;
            
            % Using whole_ROI_grey with gaussian bandpass filter
            mn_wwfR=nanmean(gray_cs_corrected_wwfR(:));
            std_wwfR=nanstd(gray_cs_corrected_wwfR(:));
            
            % Range -Tukey's Hinges
            d0_wwfR=mn_wwfR-2*(std_wwfR); % Tukey's hinge
            d1_wwfR=mn_wwfR+2*(std_wwfR);
            
            wwfR = gaussianbpf(gray_wwfR,d0_wwfR,d1_wwfR);
            
        end
        
        %
        %% Step 12 : Applying Gaussian Bandpass filter for noise correction of firing pin impression
        %
        if fpi | wof|cct|scf
            [gray_cs_corrected_fpiR, max_fpiR, min_fpiR] = convertMat2gray(cs_corrected_fpiR);
            gray_fpiR= 255.*gray_cs_corrected_fpiR;
           
            
            % Using FP_grey with gaussian bandpass filter
            mn_fpiR=nanmean(gray_fpiR(:));
            std_fpiR=nanstd(gray_fpiR(:));
            
            %Range -16-250 um for FP - micrometer to m
            % D0_FP=mn_FP-2*(stD_FP); % Tukey's hinge
            % D1_FP=mn_FP+2*(stD_FP);
            d0_fpiR=((1/max_fpiR)*0.000016)*255;
            d1_fpiR=((1/max_fpiR)*0.000250)*255;
            
            fpiR = gaussianbpf(gray_fpiR,d0_fpiR,d1_fpiR);
        end
        
        
        %% Step 13 : Applying Gaussian Bandpass filter for noise correction of breech face region
        %
        if bfr | wof|cct|scf
            [cs_corrected_bfrR, max_bfrR, min_bfrR] = convertMat2gray(cs_corrected_bfrR);
            gray_bfrR= 255.*cs_corrected_bfrR;
            
            % Using BF_grey with gaussian bandpass filter
            mn_bfrR=nanmean(gray_bfrR(:));
            std_bfrR=nanstd(gray_bfrR(:));
            
            %Range -20-150 um for BF - micrometer to m
            % D0_BF=mn_BF-2*(stD_BF);
            % D1_BF=mn_BF+2*(stD_BF);
            d0_bfrR=((1/max_bfrR)*0.000020)*255;
            d1_bfrR=((1/max_bfrR)*0.000150)*255;
            %
            bfrR = gaussianbpf(gray_bfrR,d0_bfrR,d1_bfrR);
            
        end
        
        %% Step 14 : Fusion of separately illumination corrected breech face region and firing pin impression
        if wof
            wofR=zeros(size(bfrR));
            
            xmin_fpiR=boundingBox_fpiR(1,1);
            ymin_fpiR=boundingBox_fpiR(1,2);
            width_fpiR=boundingBox_fpiR(1,3);
            height_fpiR=boundingBox_fpiR(1,4);
            
            wofR(ymin_fpiR:ymin_fpiR+height_fpiR, xmin_fpiR:xmin_fpiR+width_fpiR)=fpiR;
            
            wofR=wofR+bfrR;
            
        end
        %% Step 15 : Reshape into a square for moment estimation - whole ROI
        %
        if wwf
            wwfR_sq=double(uint8(reshape_for_moment(wwfR))); % makes image square for fitting unit circle
            cmask_wwfR_sq=double(uint8(reshape_for_moment(cmask_wwfR))); % makes image square for fitting unit circle
        end
        %% Step 16 : Reshape into a square for moment estimation - firing pin impression
        if fpi|cct|scf
            fpiR_sq=double(uint8(reshape_for_moment(fpiR))); % makes image square for fitting unit circle
        end
        %% Step 17 : Reshape into a square for moment estimation - breech face region
        if bfr|cct|scf
            bfrR_sq=double(uint8(reshape_for_moment(bfrR))); % makes image square for fitting unit circle
        end
        %% Step 18 : Reshape into a square for moment estimation - fused image consisting of breech face + fpi w/o flowback region
        if wof
            wofR_sq=double(uint8(reshape_for_moment(wofR))); % makes image square for fitting unit circle
        end
        %
        %% Step 19 : Estimating central moments
        
        cem_wwfR=[];
        cem_fpiR=[];
        cem_bfrR=[];
        cem_wofR=[];
        
        
        if cem
            % For ROI with flowback
            if wwf
                cem_wwfR = central_Moments_fun(wwfR_sq,16);
            end
           
            % For FPI
            if fpi|cct|scf
                cem_fpiR = central_Moments_fun(fpiR_sq,7);
                
            end
           
            % For BF
            if bfr|cct|scf
                cem_bfrR = central_Moments_fun(bfrR_sq,14);
            end
            
            % For ROI without flowback
            if wof
                cem_wofR= central_Moments_fun(wofR_sq,16);
            end
            
            central_moments_feat=struct('cem_wwfR',cem_wwfR,...
                'cem_fpiR',cem_fpiR,...
                'cem_bfrR',cem_bfrR,...
                'cem_wofR',cem_wofR);
            saveMat('central_moments_feat',central_moments_feat);
        end
        
        %% Step 20 : Estimating circle moment invariants
        cim_wwfR=[];
        cim_fpiR=[];
        cim_bfrR=[];
        cim_wofR=[];
        
        
        if cim
            
            % For ROI with flowback
            if wwf
                cim_wwfR = circle_Moments_Invariants_fun(wwfR_sq,16);
            end
            
            % For FPI
            if fpi|cct | scf
                cim_fpiR = circle_Moments_Invariants_fun(fpiR_sq,7);
            end
            
            % For BF
            if bfr|cct|scf
                cim_bfrR = circle_Moments_Invariants_fun(bfrR_sq,14);
            end
            
            % For ROI without flowback
            if wof
                cim_wofR = circle_Moments_Invariants_fun(wofR_sq,16);
                %
            end
            
            circle_momentsIv_feat=struct('cim_wwfR',cim_wwfR,...
                'cim_fpiR',cim_fpiR,...
                'cim_bfrR',cim_bfrR,...
                'cim_wofR',cim_wofR);
            saveMat('circle_momentsIv_feat',circle_momentsIv_feat);
        end
        
        %% Step 21 : Estimating central moment invariants
        
        cmi_wwfR=[];
        cmi_fpiR=[];
        cmi_bfrR=[];
        cmi_wofR=[];
        
        
        if cmi
            % For ROI with flowback
            if wwf
                cmi_wwfR = central_Moments_Invariants_all(wwfR_sq,16);
            end
          
            % For FPI
            if fpi|cct|scf
                cmi_fpiR = central_Moments_Invariants_all(fpiR_sq,7);
            end
           
            % For BF
            if bfr|cct|scf
                cmi_bfrR = central_Moments_Invariants_all(bfrR_sq,14);
            end
            %
            % For ROI without flowback
            if wof
                cmi_wofR = central_Moments_Invariants_all(wofR_sq,16);
            end
            
            central_momentsIv_feat=struct('cmi_wwfR',cmi_wwfR,...
                'cmi_fpiR',cmi_fpiR,...
                'cmi_bfrR',cmi_bfrR,...
                'cmi_wofR',cmi_wofR);
            saveMat('central_momentsIv_feat',central_momentsIv_feat);
        end
        
        %% Step 22 : Legendre moments
        
        lmi_wwfR=[];
        lmi_fpiR=[];
        lmi_bfrR=[];
        lmi_wofR=[];
        
        
        
        
        if lmi
           
            % For ROI with flowback
            if wwf
                lmi_wwfR = legendre_Moments_fun(wwfR_sq,16);
                
            end
            
            % For FP
            if fpi|cct|scf
                lmi_fpiR = legendre_Moments_fun(fpiR_sq,7);
            end
            
            % For BF
            if bfr|cct|scf
                lmi_bfrR = legendre_Moments_fun(bfrR_sq,14);
            end
            
            % For ROI without flowback
            if wof
                lmi_wofR = legendre_Moments_fun(wofR_sq,16);
            end
            
            legendre_moments_feat=struct('lmi_wwfR',lmi_wwfR,...
                'lmi_fpiR',lmi_fpiR,...
                'lmi_bfrR',lmi_bfrR,...
                'lmi_wofR',lmi_wofR);
            saveMat('legendre_moments_feat',legendre_moments_feat);
        end
        
        %% Step 23 : Fourier transform moments
               
        fma_wwfR=[];
               
          
        if fma
            
            if wwf
                fma_wwfR= circle_fourier_transform(wwfR_sq,cmask_wwfR_sq,max_radius,23,1);
               
            end
            
            fourier_moments_m_feat=struct('fma_wwfR',fma_wwfR);
                
            saveMat('fourier_moments_m_feat',fourier_moments_m_feat);
        end
        
        
        fmp_wwfR=[];
               
        if fmp
            
            if wwf
               fmp_wwfR = circle_fourier_transform(wwfR_sq,cmask_wwfR_sq,max_radius,23,2);
            end
            
            
            fourier_moments_mp_feat=struct('fmp_wwfR',fmp_wwfR);
                
            saveMat('fourier_moments_mp_feat',fourier_moments_mp_feat);
        end
        
        %% Step 24 : Zernike moments
        
        
        zma_wwfR=[];
        zma_fpiR=[];
        zma_bfrR=[];
        zma_wofR=[];
        
        
        
        if zma
            if wwf
                zma_wwfR = zernike_moments(wwfR_sq,23,1);
                
            end   
       
            if fpi|cct|scf
                zma_fpiR = zernike_moments(fpiR_sq,10,1);
                
            end
            
            if bfr|cct|scf
                zma_bfrR = zernike_moments(bfrR_sq,20,1);
            
            end
            
            if wof
                zma_wofR = zernike_moments(wofR_sq,23,1);
      
            end
            
            zernike_moments_m_feat=struct('zma_wwfR',zma_wwfR,...
                'zma_fpiR',zma_fpiR,...
                'zma_bfrR',zma_bfrR,...
                'zma_wofR',zma_wofR);
            
            saveMat('zernike_moments_m_feat',zernike_moments_m_feat);
        end
       
        zmp_wwfR=[];
        zmp_fpiR=[];
        zmp_bfrR=[];
        zmp_wofR=[];
        
        if zmp
            if wwf
               zmp_wwfR = zernike_moments(wwfR_sq,23,2);
                
            end   
               
            if fpi|cct|scf
                zmp_fpiR = zernike_moments(fpiR_sq,10,2);
            
            end
            
            if bfr|cct|scf
                zmp_bfrR = zernike_moments(bfrR_sq,20,2);
          
            end
            
            if wof
                zmp_wofR = zernike_moments(wofR_sq,23,2);
           
            end
            
            zernike_moments_mp_feat=struct('zmp_wwfR',zmp_wwfR,...
                'zmp_fpiR',zmp_fpiR,...
                'zmp_bfrR',zmp_bfrR,...
                'zmp_wofR',zmp_wofR);
         
            
            saveMat('zernike_moments_mp_feat',zernike_moments_mp_feat);
            
            
            
            
        end
        
        
    %end
    cd(work_folder);
    
    
    
end

clear all;

work_folder='E:\Cartridge Case Project\';

%% find foldernames of relevance (i.e. folders having same name as the cartridge case sample) using regular expression

% Get a list of all files and folders in this folder.
files = dir(work_folder);

% Get a logical vector that identify directory.
dirFlags = [files.isdir];

% Extract only those that are directories.
subFolders = files(dirFlags);

cd(work_folder);

relevant_folders=[];
% Among folders find folders whose name has a given syntax
for i = 1 : length(subFolders)
    
    folderName = subFolders(i).name;
    
    % select relevant folders using regexp
    if select_im_folders(folderName)
        relevant_folders=[relevant_folders; convertCharsToStrings(folderName)];
    end
    
end

cd(work_folder);
load('flags.mat');

% initialize matrices
cem_wwfR=[]; cem_fpiR=[];cem_bfrR=[]; cem_wofR=[];
cim_wwfR=[]; cim_fpiR=[];cim_bfrR=[]; cim_wofR=[];
cmi_wwfR=[]; cmi_fpiR=[];cmi_bfrR=[]; cmi_wofR=[];
lmi_wwfR=[]; lmi_fpiR=[];lmi_bfrR=[]; lmi_wofR=[];
fma_wwfR=[]; fmp_wwfR=[];
zma_wwfR=[];zma_fpiR=[];zma_bfrR=[]; zma_wofR=[];
zmp_wwfR=[];zmp_fpiR=[];zmp_bfrR=[]; zmp_wofR=[];
labs=[];

for i=1:length(relevant_folders)
    path = strcat(work_folder,relevant_folders(i),'\');
    cd(char(path));
    
    if cem
        load central_moments_feat.mat
        
        if wwf
            cem_wwfR(i,:)=DataI.cem_wwfR;
            
        end
        
        if fpi|cct|scf
            cem_fpiR(i,:)=DataI.cem_fpiR;
            
        end
        
        if bfr|cct|scf
            cem_bfrR(i,:)=DataI.cem_bfrR;
            
        end
        
        if wof
            cem_wofR(i,:)=DataI.cem_wofR;
            
        end
    end
    
    if cim
        load circle_momentsIv_feat.mat
        
        if wwf
            cim_wwfR(i,:)=DataI.cim_wwfR;
            
        end
        
        if fpi|cct|scf
            cim_fpiR(i,:)=DataI.cim_fpiR;
            
        end
        
        if bfr|cct|scf
            cim_bfrR(i,:)=DataI.cim_bfrR;
            
        end
        
        if wof
            cim_wofR(i,:)=DataI.cim_wofR;
            
        end
    end
    
    if cmi
        load central_momentsIv_feat.mat
        
        if wwf
            cmi_wwfR(i,:)=DataI.cmi_wwfR;
            
        end
        
        if fpi|cct|scf
            cmi_fpiR(i,:)=DataI.cmi_fpiR;
            
        end
        
        if bfr|cct|scf
            cmi_bfrR(i,:)=DataI.cmi_bfrR;
            
        end
        
        if wof
            cmi_wofR(i,:)=DataI.cmi_wofR;
            
        end
    end
    
    if lmi
        load legendre_moments_feat.mat
        
        if wwf
            lmi_wwfR(i,:)=DataI.lmi_wwfR;
            
        end
        
        if fpi|cct|scf
            lmi_fpiR(i,:)=DataI.lmi_fpiR;
            
        end
        
        if bfr|cct|scf
            lmi_bfrR(i,:)=DataI.lmi_bfrR;
            
        end
        
        if wof
            lmi_wofR(i,:)=DataI.lmi_wofR;
            
        end
    end
    
    if fma
        load fourier_moments_m_feat.mat
        if wwf
            fma_wwfR(i,:)=DataI.fma_wwfR;
            
        end
    end
        
    if fmp
        load fourier_moments_mp_feat.mat
        if wwf
            fmp_wwfR(i,:)=DataI.fmp_wwfR;
            
        end
    end
    
    
    if zma
        load zernike_moments_m_feat.mat
        
        if wwf
            zma_wwfR(i,:)=DataI.zma_wwfR;
        end
        
        if fpi|cct|scf
            zma_fpiR(i,:)=DataI.zma_fpiR;
        end
        
        if bfr|cct|scf
            zma_bfrR(i,:)=DataI.zma_bfrR;
        end
        
        if wof
            zma_wofR(i,:)=DataI.zma_wofR;
        end
    end
    
    if zmp
        load zernike_moments_mp_feat.mat
        
        if wwf
            zmp_wwfR(i,:)=DataI.zmp_wwfR;
        end
        
        if fpi|cct|scf
            zmp_fpiR(i,:)=DataI.zmp_fpiR;
        end
        
        if bfr|cct|scf
            zmp_bfrR(i,:)=DataI.zmp_bfrR;
        end
        
        if wof
            zmp_wofR(i,:)=DataI.zmp_wofR;
        end
    end
    
    %% Generate List of labels
    load details.mat
    labs(i,1)=DataI.lab;
    
    cd(work_folder);
end

%% This is the 2nd part
rng(0);
%% Step 1 : Select guns that will be used for further processing
uLab=unique(labs);

instances_eachGun=[];guns_to_consider=[];
for i=1:length(uLab)
    c=0;
    for j=1:length(labs)
        if labs(j)==uLab(i)
            c=c+1;
            instances_eachGun(i)=c;
        end
    end
    if instances_eachGun(i)>=8   % consider only guns that have greater than or equal to 8 cartridge case scans
        guns_to_consider=[guns_to_consider uLab(i)];
    end
end

%% Step 2 : Divide Gun_to_consider into train and test sets in the ratio 0.66:0.33
[an1, bn1, cn1]=mkdir('D:\','Results');

folder_to_store='D:\Results\';

ratio_training =2/3;
% Guns to reject
to_remove=[1, 111, 114, 115, 117, 118, 126, 131, 133, ...
    134, 136, 139, 145, 153, 204, 208, 214, 280,...
    283, 289, 290, 291, 306, 310, 320,32, 34, 40, 108,...
    109, 110, 112, 113, 116, 123, 125, 127, 138, 154, 168, 202];

guns_to_consider=setdiff( guns_to_consider,to_remove);
uLab_idx=randperm(length(guns_to_consider));
uLab_rand=guns_to_consider(uLab_idx);
train_no_of_diff_guns=round(ratio_training*length(uLab_rand)); %

%Gun numbers used for training
gun_nos_train=uLab_rand(1:train_no_of_diff_guns);

% training indices
idx=[];
for j=1:length(gun_nos_train)
    idxn=(labs==gun_nos_train(j));
    set=1:length(idxn);
    eachGun=set(idxn==1);
    idx=[idx;eachGun'];
end

% Training Labels
train_labels=labs(idx);

% Enrolment and test indices
gun_nos_enrol_test=uLab_rand(train_no_of_diff_guns+1:end);

idxET=[];
for j=1:length(gun_nos_enrol_test)
    idxE=(labs==gun_nos_enrol_test(j));
    setE=1:length(idxE);
    eachGunE=setE(idxE==1);
    idxET=[idxET;eachGunE'];
end

% Enrolment and test Labels
entest_labels=labs(idxET,:);

numIter=20;

if cem
    if wwf
        % train data
        train_cem_wwfR=cem_wwfR(idx,:);
                
        % enrol & test data
        entest_cem_wwfR=cem_wwfR(idxET,:);
                
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_cem_wwfR_pls,entest_cem_wwfR_pls] = pca_lda_scale(train_cem_wwfR,train_labels,entest_cem_wwfR,1);
        
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cem_wwfR_model_1v1 = plda_twoCov_train22(numIter,train_cem_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_wwfR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(cem_wwfR_model_1v1, entest_cem_wwfR_pls,entest_labels,opt);
            
            
            % calibration & validation
            % flag set to 1 to plot tippett
            cllr_cem_wwfR_1v1 = calibration_validation(cem_wwfR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'cem wwfR 1v1');
            
            
        end
    
        % 1v3
        if do_1v3
            opt=2;
            
            % plda train
            cem_wwfR_model_1v3 = plda_twoCov_train22(numIter,train_cem_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_wwfR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(cem_wwfR_model_1v3, entest_cem_wwfR_pls,entest_labels,opt);
            
            % calibration & validation
            % flag set to 1 to plot tippett
            cllr_cem_wwfR_1v3 = calibration_validation(cem_wwfR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'cem wwfR 1v3');
            
        end
        
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cem_wwfR_model_1v9 = plda_twoCov_train22(numIter,train_cem_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_wwfR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(cem_wwfR_model_1v9, entest_cem_wwfR_pls,entest_labels,opt);
            
            
            cllr_cem_wwfR_1v9 = calibration_validation(cem_wwfR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'cem wwfR 1v9');
            
        end
        
    end
    
    if fpi|scf
        % train data
        train_cem_fpiR=cem_fpiR(idx,:);
        
        
        % enrol & test data
        entest_cem_fpiR=cem_fpiR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=2, 10 PCA dim selected
        [train_cem_fpiR_pls,entest_cem_fpiR_pls] = pca_lda_scale(train_cem_fpiR,train_labels,entest_cem_fpiR,2);
        
        
        
        % 1v1
        if do_1v1
            opt=1;
            
            % plda train
            cem_fpiR_model_1v1 = plda_twoCov_train22(numIter,train_cem_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_fpiR_plda_scores_1v1, lab_test_fpiR_1v1, gun_pairs_fpiR_1v1] = plda_twoCov_test(cem_fpiR_model_1v1, entest_cem_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_fpiR_1v1 = calibration_validation(cem_fpiR_plda_scores_1v1,lab_test_fpiR_1v1,gun_pairs_fpiR_1v1,folder_to_store,1,'cem fpiR 1v1');
            
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            % plda train
            cem_fpiR_model_1v3 = plda_twoCov_train22(numIter,train_cem_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_fpiR_plda_scores_1v3, lab_test_fpiR_1v3, gun_pairs_fpiR_1v3] = plda_twoCov_test(cem_fpiR_model_1v3, entest_cem_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_fpiR_1v3 = calibration_validation(cem_fpiR_plda_scores_1v3,lab_test_fpiR_1v3,gun_pairs_fpiR_1v3,folder_to_store,1,'cem fpiR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cem_fpiR_model_1v9 = plda_twoCov_train22(numIter,train_cem_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_fpiR_plda_scores_1v9, lab_test_fpiR_1v9, gun_pairs_fpiR_1v9] = plda_twoCov_test(cem_fpiR_model_1v9, entest_cem_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_fpiR_1v9 = calibration_validation(cem_fpiR_plda_scores_1v9,lab_test_fpiR_1v9,gun_pairs_fpiR_1v9,folder_to_store,1,'cem fpiR 1v9');
            
        end
    end
    
    if bfr|scf
        % train data
        train_cem_bfrR=cem_bfrR(idx,:);
        
        
        % enrol & test data
        entest_cem_bfrR=cem_bfrR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=3, 20 PCA dim selected
        [train_cem_bfrR_pls,entest_cem_bfrR_pls] = pca_lda_scale(train_cem_bfrR,train_labels,entest_cem_bfrR,3);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cem_bfrR_model_1v1 = plda_twoCov_train22(numIter,train_cem_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_bfrR_plda_scores_1v1, lab_test_bfrR_1v1, gun_pairs_bfrR_1v1] = plda_twoCov_test(cem_bfrR_model_1v1, entest_cem_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_bfrR_1v1 = calibration_validation(cem_bfrR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'cem bfrR 1v1');
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            % plda train
            cem_bfrR_model_1v3 = plda_twoCov_train22(numIter,train_cem_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_bfrR_plda_scores_1v3, lab_test_bfrR_1v3, gun_pairs_bfrR_1v3] = plda_twoCov_test(cem_bfrR_model_1v3, entest_cem_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_bfrR_1v3 = calibration_validation(cem_bfrR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'cem bfrR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cem_bfrR_model_1v9 = plda_twoCov_train22(numIter,train_cem_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_bfrR_plda_scores_1v9, lab_test_bfrR_1v9, gun_pairs_bfrR_1v9] = plda_twoCov_test(cem_bfrR_model_1v9, entest_cem_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_bfrR_1v9 = calibration_validation(cem_bfrR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9,folder_to_store,1,'cem bfrR 1v9');
            
        end
    end
    
    if wof
        % train data
        train_cem_wofR=cem_wofR(idx,:);
        
        
        % enrol & test data
        entest_cem_wofR=cem_wofR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_cem_wofR_pls,entest_cem_wofR_pls] = pca_lda_scale(train_cem_wofR,train_labels,entest_cem_wofR,1);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cem_wofR_model_1v1 = plda_twoCov_train22(numIter,train_cem_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_wofR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(cem_wofR_model_1v1, entest_cem_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_wofR_1v1 = calibration_validation(cem_wofR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'cem wofR 1v1');
            
            
        end
        
       % 1v3
        if do_1v3
            opt=2;
            % plda train
            cem_wofR_model_1v3 = plda_twoCov_train22(numIter,train_cem_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_wofR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(cem_wofR_model_1v3, entest_cem_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_wofR_1v3 = calibration_validation(cem_wofR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'cem wofR 1v3');
            
            
        end
        
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cem_wofR_model_1v9 = plda_twoCov_train22(numIter,train_cem_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_wofR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(cem_wofR_model_1v9, entest_cem_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_wofR_1v9 = calibration_validation(cem_wofR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'cem wofR 1v9');
            
        end
    
    end
    if  cct    % feature level fusion
        % train data
        train_cem_cctR=[cem_bfrR(idx,:), cem_fpiR(idx,:)];
        
        
        % enrol & test data
        entest_cem_cctR=[cem_bfrR(idxET,:) cem_fpiR(idxET,:)];
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_cem_cctR_pls,entest_cem_cctR_pls] = pca_lda_scale(train_cem_cctR,train_labels,entest_cem_cctR,1);
        
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cem_cctR_model_1v1 = plda_twoCov_train22(numIter,train_cem_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_cctR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(cem_cctR_model_1v1, entest_cem_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_cctR_1v1 = calibration_validation(cem_cctR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'cem cctR 1v1');
            
        end
        
         % 1v3
        if do_1v3
            opt=2;
            % plda train
            cem_cctR_model_1v3 = plda_twoCov_train22(numIter,train_cem_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_cctR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(cem_cctR_model_1v3, entest_cem_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_cctR_1v3 = calibration_validation(cem_cctR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'cem cctR 1v3');
            
            
        end
        
               
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cem_cctR_model_1v9 = plda_twoCov_train22(numIter,train_cem_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [cem_cctR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(cem_cctR_model_1v9, entest_cem_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cem_cctR_1v9 = calibration_validation(cem_cctR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'cem cctR 1v9');
            
        end
    end
    
    if scf
        
        if do_1v1
            % for 1v1
            cem_scfR_plda_scores_1v1= [cem_bfrR_plda_scores_1v1 cem_fpiR_plda_scores_1v1];
            
            
            cllr_cem_scfR_1v1= calibration_validation(cem_scfR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'cem scfR 1v1');
            
        end
        
        if do_1v3
            % for 1v3
            cem_scfR_plda_scores_1v3=[cem_bfrR_plda_scores_1v3 cem_fpiR_plda_scores_1v3];
            
            
            cllr_cem_scfR_1v3= calibration_validation(cem_scfR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'cem scfR 1v3');
            
        end
        
        if do_1v9
            % for 1v9
            cem_scfR_plda_scores_1v9=[cem_bfrR_plda_scores_1v9 cem_fpiR_plda_scores_1v9];
            
            
            cllr_cem_scfR_1v9= calibration_validation(cem_scfR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9,folder_to_store,1,'cem scfR 1v9');
            
        end
    
    end
    
end

if cim
    if wwf
        % train data
        train_cim_wwfR=cim_wwfR(idx,:);
        
        
        % enrol & test data
        entest_cim_wwfR=cim_wwfR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_cim_wwfR_pls,entest_cim_wwfR_pls] = pca_lda_scale(train_cim_wwfR,train_labels,entest_cim_wwfR,1);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cim_wwfR_model_1v1 = plda_twoCov_train22(numIter,train_cim_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_wwfR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(cim_wwfR_model_1v1, entest_cim_wwfR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_wwfR_1v1 = calibration_validation(cim_wwfR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'cim wwfR 1v1');
      
        end
        
           % 1v3
        if do_1v3
            opt=2;
            
            % plda train
            cim_wwfR_model_1v3 = plda_twoCov_train22(numIter,train_cim_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_wwfR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(cim_wwfR_model_1v3, entest_cim_wwfR_pls,entest_labels,opt);
            
            
            cllr_cim_wwfR_1v3 = calibration_validation(cim_wwfR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'cim wwfR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cim_wwfR_model_1v9 = plda_twoCov_train22(numIter,train_cim_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_wwfR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(cim_wwfR_model_1v9, entest_cim_wwfR_pls,entest_labels,opt);
            
            
            cllr_cim_wwfR_1v9= calibration_validation(cim_wwfR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'cim wwfR 1v9');
            
        end
   end
    
    if fpi|scf
        % train data
        train_cim_fpiR=cim_fpiR(idx,:);
        
        
        % enrol & test data
        entest_cim_fpiR=cim_fpiR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=2, 10 PCA dim selected
        [train_cim_fpiR_pls,entest_cim_fpiR_pls] = pca_lda_scale(train_cim_fpiR,train_labels,entest_cim_fpiR,2);
        
        
        % 1v1
        if do_1v1
            opt=1;
            
            % plda train
            cim_fpiR_model_1v1 = plda_twoCov_train22(numIter,train_cim_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_fpiR_plda_scores_1v1, lab_test_fpiR_1v1, gun_pairs_fpiR_1v1] = plda_twoCov_test(cim_fpiR_model_1v1, entest_cim_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_fpiR_1v1 = calibration_validation(cim_fpiR_plda_scores_1v1,lab_test_fpiR_1v1,gun_pairs_fpiR_1v1,folder_to_store,1,'cim fpiR 1v1');
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            % plda train
            cim_fpiR_model_1v3 = plda_twoCov_train22(numIter,train_cim_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_fpiR_plda_scores_1v3, lab_test_fpiR_1v3, gun_pairs_fpiR_1v3] = plda_twoCov_test(cim_fpiR_model_1v3, entest_cim_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_fpiR_1v3 = calibration_validation(cim_fpiR_plda_scores_1v3,lab_test_fpiR_1v3,gun_pairs_fpiR_1v3,folder_to_store,1,'cim fpiR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cim_fpiR_model_1v9 = plda_twoCov_train22(numIter,train_cim_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_fpiR_plda_scores_1v9, lab_test_fpiR_1v9, gun_pairs_fpiR_1v9] = plda_twoCov_test(cim_fpiR_model_1v9, entest_cim_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_fpiR_1v9 = calibration_validation(cim_fpiR_plda_scores_1v9,lab_test_fpiR_1v9,gun_pairs_fpiR_1v9,folder_to_store,1,'cim fpiR 1v9');
            
        end
        
    end
    if bfr|scf
        % train_data
        train_cim_bfrR=cim_bfrR(idx,:);
        
        
        % enrol & test data
        entest_cim_bfrR=cim_bfrR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=3, 20 PCA dim selected
        [train_cim_bfrR_pls,entest_cim_bfrR_pls] = pca_lda_scale(train_cim_bfrR,train_labels,entest_cim_bfrR,3);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cim_bfrR_model_1v1 = plda_twoCov_train22(numIter,train_cim_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_bfrR_plda_scores_1v1, lab_test_bfrR_1v1, gun_pairs_bfrR_1v1] = plda_twoCov_test(cim_bfrR_model_1v1, entest_cim_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_bfrR_1v1 = calibration_validation(cim_bfrR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'cim bfrR 1v1');
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            % plda train
            cim_bfrR_model_1v3 = plda_twoCov_train22(numIter,train_cim_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_bfrR_plda_scores_1v3, lab_test_bfrR_1v3, gun_pairs_bfrR_1v3] = plda_twoCov_test(cim_bfrR_model_1v3, entest_cim_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_bfrR_1v3 = calibration_validation(cim_bfrR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'cim bfrR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cim_bfrR_model_1v9 = plda_twoCov_train22(numIter,train_cim_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_bfrR_plda_scores_1v9, lab_test_bfrR_1v9, gun_pairs_bfrR_1v9] = plda_twoCov_test(cim_bfrR_model_1v9, entest_cim_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_bfrR_1v9 = calibration_validation(cim_bfrR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9,folder_to_store,1,'cim bfrR 1v9');
            
        end
        
        
    end
    if wof
        % train data
        train_cim_wofR=cim_wofR(idx,:);
        
        
        % enrol & test data
        entest_cim_wofR=cim_wofR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_cim_wofR_pls,entest_cim_wofR_pls] = pca_lda_scale(train_cim_wofR,train_labels,entest_cim_wofR,1);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cim_wofR_model_1v1 = plda_twoCov_train22(numIter,train_cim_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_wofR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(cim_wofR_model_1v1, entest_cim_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_wofR_1v1 = calibration_validation(cim_wofR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'cim wofR 1v1');
            
            
        end
        
          % 1v3
        if do_1v3
            opt=2;
            % plda train
            cim_wofR_model_1v3 = plda_twoCov_train22(numIter,train_cim_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_wofR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(cim_wofR_model_1v3, entest_cim_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_wofR_1v3 = calibration_validation(cim_wofR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'cim wofR 1v3');
            
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cim_wofR_model_1v9 = plda_twoCov_train22(numIter,train_cim_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_wofR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(cim_wofR_model_1v9, entest_cim_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_wofR_1v9 = calibration_validation(cim_wofR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'cim wofR 1v9');
            
            
        end
        
      
    end
    if  cct    % feature level fusion
        % train data
        train_cim_cctR=[cim_bfrR(idx,:) cim_fpiR(idx,:)];
        
        
        % enrol & test data
        entest_cim_cctR=[cim_bfrR(idxET,:) cim_fpiR(idxET,:)];
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_cim_cctR_pls,entest_cim_cctR_pls] = pca_lda_scale(train_cim_cctR,train_labels,entest_cim_cctR,1);
        
     
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cim_cctR_model_1v1 = plda_twoCov_train22(numIter,train_cim_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_cctR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(cim_cctR_model_1v1, entest_cim_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_cctR_1v1 = calibration_validation(cim_cctR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'cim cctR 1v1');
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            % plda train
            cim_cctR_model_1v3 = plda_twoCov_train22(numIter,train_cim_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_cctR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(cim_cctR_model_1v3, entest_cim_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_cctR_1v3 = calibration_validation(cim_cctR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'cim cctR 1v3');
            
            
        end
        
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cim_cctR_model_1v9 = plda_twoCov_train22(numIter,train_cim_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [cim_cctR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(cim_cctR_model_1v9, entest_cim_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cim_cctR_1v9 = calibration_validation(cim_cctR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'cim cctR 1v9');
            
        end
        
  end
    
    if scf
        
        if do_1v1
            % for 1v1
            cim_scfR_plda_scores_1v1= [cim_bfrR_plda_scores_1v1 cim_fpiR_plda_scores_1v1];
            
            
            cllr_cim_scfR_1v1= calibration_validation(cim_scfR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'cim scfR 1v1');
            
        end
        
        if do_1v3
            % for 1v3
            cim_scfR_plda_scores_1v3=[cim_bfrR_plda_scores_1v3 cim_fpiR_plda_scores_1v3];
            
            
            cllr_cim_scfR_1v3= calibration_validation(cim_scfR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'cim scfR 1v3');
            
        end
        
        if do_1v9
            % for 1v9
            cim_scfR_plda_scores_1v9=[cim_bfrR_plda_scores_1v9 cim_fpiR_plda_scores_1v9];
            
            
            cllr_cim_scfR_1v9= calibration_validation(cim_scfR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9,folder_to_store,1,'cim scfR 1v9');
            
        end
    end
    
end

if cmi
    if wwf
        train_cmi_wwfR=cmi_wwfR(idx,:);
        
        
        entest_cmi_wwfR=cmi_wwfR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_cmi_wwfR_pls,entest_cmi_wwfR_pls] = pca_lda_scale(train_cmi_wwfR,train_labels,entest_cmi_wwfR,1);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cmi_wwfR_model_1v1 = plda_twoCov_train22(numIter,train_cmi_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_wwfR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(cmi_wwfR_model_1v1, entest_cmi_wwfR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_wwfR_1v1 = calibration_validation(cmi_wwfR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'cmi wwfR 1v1');
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            
            % plda train
            cmi_wwfR_model_1v3 = plda_twoCov_train22(numIter,train_cmi_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_wwfR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(cmi_wwfR_model_1v3, entest_cmi_wwfR_pls,entest_labels,opt);
            
            
            cllr_cmi_wwfR_1v3 = calibration_validation(cmi_wwfR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'cmi wwfR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cmi_wwfR_model_1v9 = plda_twoCov_train22(numIter,train_cmi_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_wwfR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(cmi_wwfR_model_1v9, entest_cmi_wwfR_pls,entest_labels,opt);
            
            
            cllr_cmi_wwfR_1v9 = calibration_validation(cmi_wwfR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'cmi wwfR 1v9');
            
        end
   end
    
    
    if fpi|scf
        train_cmi_fpiR=cmi_fpiR(idx,:);
        
        
        entest_cmi_fpiR=cmi_fpiR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=2, 10 PCA dim selected
        [train_cmi_fpiR_pls,entest_cmi_fpiR_pls] = pca_lda_scale(train_cmi_fpiR,train_labels,entest_cmi_fpiR,2);
        
        
        % 1v1
        if do_1v1
            opt=1;
            
            % plda train
            cmi_fpiR_model_1v1 = plda_twoCov_train22(numIter,train_cmi_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_fpiR_plda_scores_1v1, lab_test_fpiR_1v1, gun_pairs_fpiR_1v1] = plda_twoCov_test(cmi_fpiR_model_1v1, entest_cmi_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_fpiR_1v1 = calibration_validation(cmi_fpiR_plda_scores_1v1,lab_test_fpiR_1v1,gun_pairs_fpiR_1v1,folder_to_store,1,'cmi fpiR 1v1');
            
        end
        
         % 1v3
        if do_1v3
            opt=2;
            % plda train
            cmi_fpiR_model_1v3 = plda_twoCov_train22(numIter,train_cmi_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_fpiR_plda_scores_1v3, lab_test_fpiR_1v3, gun_pairs_fpiR_1v3] = plda_twoCov_test(cmi_fpiR_model_1v3, entest_cmi_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_fpiR_1v3 = calibration_validation(cmi_fpiR_plda_scores_1v3,lab_test_fpiR_1v3,gun_pairs_fpiR_1v3,folder_to_store,1,'cmi fpiR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cmi_fpiR_model_1v9 = plda_twoCov_train22(numIter,train_cmi_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_fpiR_plda_scores_1v9, lab_test_fpiR_1v9, gun_pairs_fpiR_1v9] = plda_twoCov_test(cmi_fpiR_model_1v9, entest_cmi_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_fpiR_1v9 = calibration_validation(cmi_fpiR_plda_scores_1v9,lab_test_fpiR_1v9,gun_pairs_fpiR_1v9,folder_to_store,1,'cmi fpiR 1v9');
            
        end
       
    end
    if bfr|scf
        train_cmi_bfrR=cmi_bfrR(idx,:);
        
        
        entest_cmi_bfrR=cmi_bfrR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=3, 20 PCA dim selected
        [train_cmi_bfrR_pls,entest_cmi_bfrR_pls] = pca_lda_scale(train_cmi_bfrR,train_labels,entest_cmi_bfrR,3);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cmi_bfrR_model_1v1 = plda_twoCov_train22(numIter,train_cmi_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_bfrR_plda_scores_1v1, lab_test_bfrR_1v1, gun_pairs_bfrR_1v1] = plda_twoCov_test(cmi_bfrR_model_1v1, entest_cmi_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_bfrR_1v1 = calibration_validation(cmi_bfrR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'cmi bfrR 1v1');
            
        end
        
         % 1v3
        if do_1v3
            opt=2;
            % plda train
            cmi_bfrR_model_1v3 = plda_twoCov_train22(numIter,train_cmi_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_bfrR_plda_scores_1v3, lab_test_bfrR_1v3, gun_pairs_bfrR_1v3] = plda_twoCov_test(cmi_bfrR_model_1v3, entest_cmi_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_bfrR_1v3 = calibration_validation(cmi_bfrR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'cmi bfrR 1v3');
            
        end
        
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cmi_bfrR_model_1v9 = plda_twoCov_train22(numIter,train_cmi_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_bfrR_plda_scores_1v9, lab_test_bfrR_1v9, gun_pairs_bfrR_1v9] = plda_twoCov_test(cmi_bfrR_model_1v9, entest_cmi_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_bfrR_1v9 = calibration_validation(cmi_bfrR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9,folder_to_store,1,'cmi bfrR 1v9');
            
        end
    end
    if wof
        train_cmi_wofR=cmi_wofR(idx,:);
        
        
        entest_cmi_wofR=cmi_wofR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_cmi_wofR_pls,entest_cmi_wofR_pls] = pca_lda_scale(train_cmi_wofR,train_labels,entest_cmi_wofR,1);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cmi_wofR_model_1v1 = plda_twoCov_train22(numIter,train_cmi_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_wofR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(cmi_wofR_model_1v1, entest_cmi_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_wofR_1v1 = calibration_validation(cmi_wofR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'cmi wofR 1v1');
            
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            % plda train
            cmi_wofR_model_1v3 = plda_twoCov_train22(numIter,train_cmi_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_wofR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(cmi_wofR_model_1v3, entest_cmi_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_wofR_1v3 = calibration_validation(cmi_wofR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'cmi wofR 1v3');
            
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cmi_wofR_model_1v9 = plda_twoCov_train22(numIter,train_cmi_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_wofR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(cmi_wofR_model_1v9, entest_cmi_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_wofR_1v9 = calibration_validation(cmi_wofR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'cmi wofR 1v9');
       
        end
    end
    if  cct    % feature level fusion
        % train data
        train_cmi_cctR=[cmi_bfrR(idx,:) cmi_fpiR(idx,:)];
        
        
        % enrol & test data
        entest_cmi_cctR=[cmi_bfrR(idxET,:) cmi_fpiR(idxET,:)];
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_cmi_cctR_pls,entest_cmi_cctR_pls] = pca_lda_scale(train_cmi_cctR,train_labels,entest_cmi_cctR,1);
        
        
        %
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            cmi_cctR_model_1v1 = plda_twoCov_train22(numIter,train_cmi_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_cctR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(cmi_cctR_model_1v1, entest_cmi_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_cctR_1v1 = calibration_validation(cmi_cctR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'cmi cctR 1v1');
            
        end
        
             % 1v3
        if do_1v3
            opt=2;
            % plda train
            cmi_cctR_model_1v3 = plda_twoCov_train22(numIter,train_cmi_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_cctR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(cmi_cctR_model_1v3, entest_cmi_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_cctR_1v3 = calibration_validation(cmi_cctR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'cmi cctR 1v3');
            
            
        end
        
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            cmi_cctR_model_1v9 = plda_twoCov_train22(numIter,train_cmi_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [cmi_cctR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(cmi_cctR_model_1v9, entest_cmi_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_cmi_cctR_1v9 = calibration_validation(cmi_cctR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'cmi cctR 1v9');
            
        end
        
        
  
        
    end
    
    if scf
        
        if do_1v1
            % for 1v1
            cmi_scfR_plda_scores_1v1= [cmi_bfrR_plda_scores_1v1 cmi_fpiR_plda_scores_1v1];
            
            
            cllr_cmi_scfR_1v1= calibration_validation(cmi_scfR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'cmi scfR 1v1');
            
        end
        
        if do_1v3
            % for 1v3
            cmi_scfR_plda_scores_1v3=[cmi_bfrR_plda_scores_1v3 cmi_fpiR_plda_scores_1v3];
            
            
            cllr_cmi_scfR_1v3= calibration_validation(cmi_scfR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'cmi scfR 1v3');
            
        end
        
        
        if do_1v9
            % for 1v9
            cmi_scfR_plda_scores_1v9=[cmi_bfrR_plda_scores_1v9 cmi_fpiR_plda_scores_1v9];
            
            
            cllr_cmi_scfR_1v9= calibration_validation(cmi_scfR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9, folder_to_store,1,'cmi scfR 1v9');
            
        end
    end
end


if lmi
    if wwf
        train_lmi_wwfR=lmi_wwfR(idx,:);
        
        
        entest_lmi_wwfR=lmi_wwfR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_lmi_wwfR_pls,entest_lmi_wwfR_pls] = pca_lda_scale(train_lmi_wwfR,train_labels,entest_lmi_wwfR,1);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            lmi_wwfR_model_1v1 = plda_twoCov_train22(numIter,train_lmi_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_wwfR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(lmi_wwfR_model_1v1, entest_lmi_wwfR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_wwfR_1v1 = calibration_validation(lmi_wwfR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'lmi wwfR 1v1');
            
            
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            
            % plda train
            lmi_wwfR_model_1v3 = plda_twoCov_train22(numIter,train_lmi_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_wwfR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(lmi_wwfR_model_1v3, entest_lmi_wwfR_pls,entest_labels,opt);
            
            
            cllr_lmi_wwfR_1v3 = calibration_validation(lmi_wwfR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'lmi wwfR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            lmi_wwfR_model_1v9 = plda_twoCov_train22(numIter,train_lmi_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_wwfR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(lmi_wwfR_model_1v9, entest_lmi_wwfR_pls,entest_labels,opt);
            
            
            cllr_lmi_wwfR_1v9 = calibration_validation(lmi_wwfR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'lmi wwfR 1v9');
            
        end
        
        
        
        
    end
    if fpi|scf
        train_lmi_fpiR=lmi_fpiR(idx,:);
        
        
        entest_lmi_fpiR=lmi_fpiR(idxET,:);
        
        
        
        % center data and estimate pca then lda and scale the data
        % Value=2, 10 PCA dim selected
        [train_lmi_fpiR_pls,entest_lmi_fpiR_pls] = pca_lda_scale(train_lmi_fpiR,train_labels,entest_lmi_fpiR,2);
        
        
        % 1v1
        if do_1v1
            opt=1;
            
            % plda train
            lmi_fpiR_model_1v1 = plda_twoCov_train22(numIter,train_lmi_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_fpiR_plda_scores_1v1, lab_test_fpiR_1v1, gun_pairs_fpiR_1v1] = plda_twoCov_test(lmi_fpiR_model_1v1, entest_lmi_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_fpiR_1v1 = calibration_validation(lmi_fpiR_plda_scores_1v1,lab_test_fpiR_1v1,gun_pairs_fpiR_1v1,folder_to_store,1,'lmi fpiR 1v1');
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            % plda train
            lmi_fpiR_model_1v3 = plda_twoCov_train22(numIter,train_lmi_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_fpiR_plda_scores_1v3, lab_test_fpiR_1v3, gun_pairs_fpiR_1v3] = plda_twoCov_test(lmi_fpiR_model_1v3, entest_lmi_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
             cllr_lmi_fpiR_1v3 = calibration_validation(lmi_fpiR_plda_scores_1v3,lab_test_fpiR_1v3,gun_pairs_fpiR_1v3,folder_to_store,1,'lmi fpiR 1v3');
            
        end
    
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            lmi_fpiR_model_1v9 = plda_twoCov_train22(numIter,train_lmi_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_fpiR_plda_scores_1v9, lab_test_fpiR_1v9, gun_pairs_fpiR_1v9] = plda_twoCov_test(lmi_fpiR_model_1v9, entest_lmi_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_fpiR_1v9 = calibration_validation(lmi_fpiR_plda_scores_1v9,lab_test_fpiR_1v9,gun_pairs_fpiR_1v9,folder_to_store,1,'lmi fpiR 1v9');
            
        end
    end   
    
    if bfr|scf
        train_lmi_bfrR=lmi_bfrR(idx,:);
        
        
        entest_lmi_bfrR=lmi_bfrR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=3, 20 PCA dim selected
        [train_lmi_bfrR_pls,entest_lmi_bfrR_pls] = pca_lda_scale(train_lmi_bfrR,train_labels,entest_lmi_bfrR,3);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            lmi_bfrR_model_1v1 = plda_twoCov_train22(numIter,train_lmi_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_bfrR_plda_scores_1v1, lab_test_bfrR_1v1, gun_pairs_bfrR_1v1] = plda_twoCov_test(lmi_bfrR_model_1v1, entest_lmi_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_bfrR_1v1 = calibration_validation(lmi_bfrR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'lmi bfrR 1v1');
            
        end
        
         % 1v3
        if do_1v3
            opt=2;
            % plda train
            lmi_bfrR_model_1v3 = plda_twoCov_train22(numIter,train_lmi_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_bfrR_plda_scores_1v3, lab_test_bfrR_1v3, gun_pairs_bfrR_1v3] = plda_twoCov_test(lmi_bfrR_model_1v3, entest_lmi_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_bfrR_1v3 = calibration_validation(lmi_bfrR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'lmi bfrR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            lmi_bfrR_model_1v9 = plda_twoCov_train22(numIter,train_lmi_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_bfrR_plda_scores_1v9, lab_test_bfrR_1v9, gun_pairs_bfrR_1v9] = plda_twoCov_test(lmi_bfrR_model_1v9, entest_lmi_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_bfrR_1v9 = calibration_validation(lmi_bfrR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9,folder_to_store,1,'lmi bfrR 1v9');
            
        end
     end
    if wof
        train_lmi_wofR=lmi_wofR(idx,:);
        
        
        entest_lmi_wofR=lmi_wofR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_lmi_wofR_pls,entest_lmi_wofR_pls] = pca_lda_scale(train_lmi_wofR,train_labels,entest_lmi_wofR,1);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            lmi_wofR_model_1v1 = plda_twoCov_train22(numIter,train_lmi_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_wofR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(lmi_wofR_model_1v1, entest_lmi_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_wofR_1v1 = calibration_validation(lmi_wofR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'lmi wofR 1v1');
            
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            % plda train
            lmi_wofR_model_1v3 = plda_twoCov_train22(numIter,train_lmi_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_wofR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(lmi_wofR_model_1v3, entest_lmi_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_wofR_1v3 = calibration_validation(lmi_wofR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'lmi wofR 1v3');
            
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            lmi_wofR_model_1v9 = plda_twoCov_train22(numIter,train_lmi_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_wofR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(lmi_wofR_model_1v9, entest_lmi_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_wofR_1v9 = calibration_validation(lmi_wofR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'lmi wofR 1v9');
     
        end
   end
    if  cct    % feature level fusion
        % train data
        train_lmi_cctR=[lmi_bfrR(idx,:) lmi_fpiR(idx,:)];
        
        
        % enrol & test data
        entest_lmi_cctR=[lmi_bfrR(idxET,:) lmi_fpiR(idxET,:)];
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_lmi_cctR_pls,entest_lmi_cctR_pls] = pca_lda_scale(train_lmi_cctR,train_labels,entest_lmi_cctR,1);
   
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            lmi_cctR_model_1v1 = plda_twoCov_train22(numIter,train_lmi_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_cctR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(lmi_cctR_model_1v1, entest_lmi_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_cctR_1v1 = calibration_validation(lmi_cctR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'lmi cctR 1v1');
            
        end
        
         % 1v3
        if do_1v3
            opt=2;
            % plda train
            lmi_cctR_model_1v3 = plda_twoCov_train22(numIter,train_lmi_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_cctR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(lmi_cctR_model_1v3, entest_lmi_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_cctR_1v3 = calibration_validation(lmi_cctR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'lmi cctR 1v3');
            
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            lmi_cctR_model_1v9 = plda_twoCov_train22(numIter,train_lmi_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [lmi_cctR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(lmi_cctR_model_1v9, entest_lmi_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_lmi_cctR_1v9 = calibration_validation(lmi_cctR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'lmi cctR 1v9');
            
        end
   end
    
    if scf
        
        if do_1v1
            % for 1v1
            lmi_scfR_plda_scores_1v1= [lmi_bfrR_plda_scores_1v1 lmi_fpiR_plda_scores_1v1];
            
            
            cllr_lmi_scfR_1v1= calibration_validation(lmi_scfR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'lmi scfR 1v1');
            
        end
        
        if do_1v3
            % for 1v3
            lmi_scfR_plda_scores_1v3=[lmi_bfrR_plda_scores_1v3 lmi_fpiR_plda_scores_1v3];
            
            
            cllr_lmi_scfR_1v3= calibration_validation(lmi_scfR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'lmi scfR 1v3');
            
        end
        
        if do_1v9
            % for 1v9
            lmi_scfR_plda_scores_1v9=[lmi_bfrR_plda_scores_1v9 lmi_fpiR_plda_scores_1v9];
            
            
            cllr_lmi_scfR_1v9= calibration_validation(lmi_scfR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9,folder_to_store,1,'lmi scfR 1v9');
            
        end
    end
end


if fma
    if wwf
        train_fma_wwfR=fma_wwfR(idx,:);
                
        entest_fma_wwfR=fma_wwfR(idxET,:);
                
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_fma_wwfR_pls,entest_fma_wwfR_pls] = pca_lda_scale(train_fma_wwfR,train_labels,entest_fma_wwfR,1);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            fma_wwfR_model_1v1 = plda_twoCov_train22(numIter,train_fma_wwfR_pls,train_labels,opt);
           
            
            %plda test scores
            [fma_wwfR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(fma_wwfR_model_1v1, entest_fma_wwfR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_fma_wwfR_1v1 = calibration_validation(fma_wwfR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'fma wwfR 1v1');
            
            
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            
            % plda train
            fma_wwfR_model_1v3 = plda_twoCov_train22(numIter,train_fma_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [fma_wwfR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(fma_wwfR_model_1v3, entest_fma_wwfR_pls,entest_labels,opt);
            
            
            cllr_fma_wwfR_1v3 = calibration_validation(fma_wwfR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'fma wwfR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            fma_wwfR_model_1v9 = plda_twoCov_train22(numIter,train_fma_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [fma_wwfR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(fma_wwfR_model_1v9, entest_fma_wwfR_pls,entest_labels,opt);
            
            
            cllr_fma_wwfR_1v9 = calibration_validation(fma_wwfR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'fma wwfR 1v9');
           
        end
    end
end
    
if fmp
    if wwf
        
        train_fmp_wwfR=fmp_wwfR(idx,:);
        
        entest_fmp_wwfR=fmp_wwfR(idxET,:);
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_fmp_wwfR_pls,entest_fmp_wwfR_pls] = pca_lda_scale(train_fmp_wwfR,train_labels,entest_fmp_wwfR,1);
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            
            fmp_wwfR_model_1v1 = plda_twoCov_train22(numIter,train_fmp_wwfR_pls,train_labels,opt);
            
            %plda test scores
            [fmp_wwfR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(fmp_wwfR_model_1v1, entest_fmp_wwfR_pls,entest_labels,opt);
            
            % calibration & validation
            cllr_fmp_wwfR_1v1 = calibration_validation(fmp_wwfR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'fmp wwfR 1v1');
            
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            
            % plda train
            fmp_wwfR_model_1v3 = plda_twoCov_train22(numIter,train_fmp_wwfR_pls,train_labels,opt);
            
            %plda test scores
            [fmp_wwfR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(fmp_wwfR_model_1v3, entest_fmp_wwfR_pls,entest_labels,opt);
            
            cllr_fmp_wwfR_1v3 = calibration_validation(fmp_wwfR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'fmp wwfR 1v3');
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            fmp_wwfR_model_1v9 = plda_twoCov_train22(numIter,train_fmp_wwfR_pls,train_labels,opt);
            
            %plda test scores
            [fmp_wwfR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(fmp_wwfR_model_1v9, entest_fmp_wwfR_pls,entest_labels,opt);
            
            cllr_fmp_wwfR_1v9 = calibration_validation(fmp_wwfR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'fmp wwfR 1v9');
        end
    end
end




if zma
    % Zernike moment orders
    % wflow/woflow- 23-total feat -156
    % fpi-10 - total feat-36
    % bf-20- total feat -121
     
    
    if wwf
        
        train_zma_wwfR=zma_wwfR(idx,:);

        
         entest_zma_wwfR=zma_wwfR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_zma_wwfR_pls,entest_zma_wwfR_pls] = pca_lda_scale(train_zma_wwfR,train_labels,entest_zma_wwfR,1);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
             zma_wwfR_model_1v1 = plda_twoCov_train22(numIter,train_zma_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_wwfR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(zma_wwfR_model_1v1, entest_zma_wwfR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_zma_wwfR_1v1 = calibration_validation(zma_wwfR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'zma wwfR 1v1');
            
            
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            
            % plda train
             zma_wwfR_model_1v3 = plda_twoCov_train22(numIter,train_zma_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_wwfR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(zma_wwfR_model_1v3, entest_zma_wwfR_pls,entest_labels,opt);
            
            cllr_zma_wwfR_1v3 = calibration_validation(zma_wwfR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'zma wwfR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
             zma_wwfR_model_1v9 = plda_twoCov_train22(numIter,train_zma_wwfR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_wwfR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(zma_wwfR_model_1v9, entest_zma_wwfR_pls,entest_labels,opt);
            
           
            cllr_zma_wwfR_1v9 = calibration_validation(zma_wwfR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'zma wwfR 1v9');
            
        end
        
    end
    if fpi|scf
     
        train_zma_fpiR=zma_fpiR(idx,:);
        
        
        entest_zma_fpiR=zma_fpiR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=2, 10 PCA dim selected
        [train_zma_fpiR_pls,entest_zma_fpiR_pls] = pca_lda_scale(train_zma_fpiR,train_labels,entest_zma_fpiR,2);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
             zma_fpiR_model_1v1 = plda_twoCov_train22(numIter,train_zma_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
             [zma_fpiR_plda_scores_1v1, lab_test_fpiR_1v1, gun_pairs_fpiR_1v1] = plda_twoCov_test(zma_fpiR_model_1v1, entest_zma_fpiR_pls,entest_labels,opt);
            
            
            % calibration & validation
             cllr_zma_fpiR_1v1 = calibration_validation(zma_fpiR_plda_scores_1v1,lab_test_fpiR_1v1,gun_pairs_fpiR_1v1,folder_to_store,1,'zma fpiR 1v1');
            
            
            
        end
        
         % 1v3
        if do_1v3
            opt=2;
            
            % plda train
             zma_fpiR_model_1v3 = plda_twoCov_train22(numIter,train_zma_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_fpiR_plda_scores_1v3, lab_test_fpiR_1v3, gun_pairs_fpiR_1v3] = plda_twoCov_test(zma_fpiR_model_1v3, entest_zma_fpiR_pls,entest_labels,opt);
            
            
            cllr_zma_fpiR_1v3 = calibration_validation(zma_fpiR_plda_scores_1v3,lab_test_fpiR_1v3, gun_pairs_fpiR_1v3,folder_to_store,1,'zma fpiR 1v3');
            
        end
        
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
             zma_fpiR_model_1v9 = plda_twoCov_train22(numIter,train_zma_fpiR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_fpiR_plda_scores_1v9, lab_test_fpiR_1v9, gun_pairs_fpiR_1v9] = plda_twoCov_test(zma_fpiR_model_1v9, entest_zma_fpiR_pls,entest_labels,opt);
            
            
            cllr_zma_fpiR_1v9 = calibration_validation(zma_fpiR_plda_scores_1v9,lab_test_fpiR_1v9,gun_pairs_fpiR_1v9,folder_to_store,1,'zma fpiR 1v9');
            
        end
        
       
        
    end
    if bfr|scf
    
        train_zma_bfrR=zma_bfrR(idx,:);
        
        
        entest_zma_bfrR=zma_bfrR(idxET,:);
        
        
        % center data and estimate pca then lda and scale the data
        % Value=3, 20 PCA dim selected
        [train_zma_bfrR_pls,entest_zma_bfrR_pls] = pca_lda_scale(train_zma_bfrR,train_labels,entest_zma_bfrR,3);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
             zma_bfrR_model_1v1 = plda_twoCov_train22(numIter,train_zma_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_bfrR_plda_scores_1v1, lab_test_bfrR_1v1, gun_pairs_bfrR_1v1] = plda_twoCov_test(zma_bfrR_model_1v1, entest_zma_bfrR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_zma_bfrR_1v1 = calibration_validation(zma_bfrR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'zma bfrR 1v1');
            
            
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            
            % plda train
             zma_bfrR_model_1v3 = plda_twoCov_train22(numIter,train_zma_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_bfrR_plda_scores_1v3, lab_test_bfrR_1v3, gun_pairs_bfrR_1v3] = plda_twoCov_test(zma_bfrR_model_1v3, entest_zma_bfrR_pls,entest_labels,opt);
            
            
            cllr_zma_bfrR_1v3 = calibration_validation(zma_bfrR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'zma bfrR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
             zma_bfrR_model_1v9 = plda_twoCov_train22(numIter,train_zma_bfrR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_bfrR_plda_scores_1v9, lab_test_bfrR_1v9, gun_pairs_bfrR_1v9] = plda_twoCov_test(zma_bfrR_model_1v9, entest_zma_bfrR_pls,entest_labels,opt);
            
            
            cllr_zma_bfrR_1v9 = calibration_validation(zma_bfrR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9,folder_to_store,1,'zma bfrR 1v9');
            
        end
        
        
        
    end
    if wof

        train_zma_wofR=zma_wofR(idx,:);
        
        
        entest_zma_wofR=zma_wofR(idxET,:);
        
        
        % Value=1, 30 PCA dim selected
        [train_zma_wofR_pls,entest_zma_wofR_pls] = pca_lda_scale(train_zma_wofR,train_labels,entest_zma_wofR,1);
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
             zma_wofR_model_1v1 = plda_twoCov_train22(numIter,train_zma_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_wofR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(zma_wofR_model_1v1, entest_zma_wofR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_zma_wofR_1v1 = calibration_validation(zma_wofR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'zma wofR 1v1');
            
            
            
        end
        
         % 1v3
        if do_1v3
            opt=2;
            
            % plda train
             zma_wofR_model_1v3 = plda_twoCov_train22(numIter,train_zma_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_wofR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(zma_wofR_model_1v3, entest_zma_wofR_pls,entest_labels,opt);
            
            
            cllr_zma_wofR_1v3 = calibration_validation(zma_wofR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'zma wofR 1v3');
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
             zma_wofR_model_1v9 = plda_twoCov_train22(numIter,train_zma_wofR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_wofR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(zma_wofR_model_1v9, entest_zma_wofR_pls,entest_labels,opt);
            
            
             cllr_zma_wofR_1v9 = calibration_validation(zma_wofR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'zma wofR 1v9');
            
        end
        
       
    end
    
    if  cct    % feature level fusion
        % train data
         train_zma_cctR=[zma_bfrR(idx,:) zma_fpiR(idx,:)];
        
        
        % enrol & test data
         entest_zma_cctR=[zma_bfrR(idxET,:) zma_fpiR(idxET,:)];
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
         [train_zma_cctR_pls,entest_zma_cctR_pls] = pca_lda_scale(train_zma_cctR,train_labels,entest_zma_cctR,1);
        
       
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
             zma_cctR_model_1v1 = plda_twoCov_train22(numIter,train_zma_cctR_pls,train_labels,opt);
            
            
            %plda test scores
             [zma_cctR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(zma_cctR_model_1v1, entest_zma_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
             cllr_zma_cctR_1v1= calibration_validation(zma_cctR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'zma cctR 1v1');
            
        end
        
         % 1v3
        if do_1v3
            opt=2;
            % plda train
             zma_cctR_model_1v3 = plda_twoCov_train22(numIter,train_zma_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_cctR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(zma_cctR_model_1v3, entest_zma_cctR_pls,entest_labels,opt);
            
            
            % calibration & validation
            cllr_zma_cctR_1v3= calibration_validation(zma_cctR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'zma cctR 1v3');
            
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
             zma_cctR_model_1v9 = plda_twoCov_train22(numIter,train_zma_cctR_pls,train_labels,opt);
            
            
            %plda test scores
            [zma_cctR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(zma_cctR_model_1v9, entest_zma_cctR_pls,entest_labels,opt);
            
            
            
            % calibration & validation
            cllr_zma_cctR_1v9= calibration_validation(zma_cctR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'zma cctR 1v9');
            
        end
        
        
       
        
    end
    
    if scf
        
        if do_1v1
            % for 1v1
            
            zma_scfR_plda_scores_1v1= [zma_bfrR_plda_scores_1v1 zma_fpiR_plda_scores_1v1];
            
            
            cllr_zma_scfR_1v1= calibration_validation(zma_scfR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'zma scfR 1v1');
            
        end
        
        if do_1v3
            % for 1v3
            
            zma_scfR_plda_scores_1v3=[zma_bfrR_plda_scores_1v3 zma_fpiR_plda_scores_1v3];
            
            
            cllr_zma_scfR_1v3= calibration_validation(zma_scfR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'zma scfR 1v3');
            
        end
        
        if do_1v9
            % for 1v9
            
            zma_scfR_plda_scores_1v9=[zma_bfrR_plda_scores_1v9 zma_fpiR_plda_scores_1v9];
            
            
            cllr_zma_scfR_1v9= calibration_validation(zma_scfR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9,folder_to_store,1,'zma scfR 1v9');
            
        end
        
        
        
    end
    
end


if zmp
    % Zernike moment orders
    % wflow/woflow- 23-total feat -300
    % fpi-10 - total feat-66
    % bf-20- total feat -231
     
    
    if wwf
        
       train_zmp_wwfR=zmp_wwfR(idx,:);
        
       entest_zmp_wwfR=zmp_wwfR(idxET,:);
        
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_zmp_wwfR_pls,entest_zmp_wwfR_pls] = pca_lda_scale(train_zmp_wwfR,train_labels,entest_zmp_wwfR,1);
                
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            zmp_wwfR_model_1v1 = plda_twoCov_train22(numIter,train_zmp_wwfR_pls,train_labels,opt);
           
            
            
            %plda test scores
            [zmp_wwfR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(zmp_wwfR_model_1v1, entest_zmp_wwfR_pls,entest_labels,opt);
            
            
            
            % calibration & validation
            cllr_zmp_wwfR_1v1 = calibration_validation(zmp_wwfR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'zmp wwfR 1v1');
            
            
            
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            
            % plda train
            zmp_wwfR_model_1v3 = plda_twoCov_train22(numIter,train_zmp_wwfR_pls,train_labels,opt);
            
            
            
            %plda test scores
            [zmp_wwfR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(zmp_wwfR_model_1v3, entest_zmp_wwfR_pls,entest_labels,opt);
            
            
            
           cllr_zmp_wwfR_1v3 = calibration_validation(zmp_wwfR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'zmp wwfR 1v3');
            
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            zmp_wwfR_model_1v9 = plda_twoCov_train22(numIter,train_zmp_wwfR_pls,train_labels,opt);
            
            
            
            %plda test scores
            [zmp_wwfR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(zmp_wwfR_model_1v9, entest_zmp_wwfR_pls,entest_labels,opt);
           
            
            
            cllr_zmp_wwfR_1v9 = calibration_validation(zmp_wwfR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'zmp wwfR 1v9');
            
            
        end
        
        
        
        
        
    end
    if fpi|scf
     
        
        
        train_zmp_fpiR=zmp_fpiR(idx,:);
        
        entest_zmp_fpiR=zmp_fpiR(idxET,:);
        
              
        % center data and estimate pca then lda and scale the data
        % Value=2, 10 PCA dim selected
        [train_zmp_fpiR_pls,entest_zmp_fpiR_pls] = pca_lda_scale(train_zmp_fpiR,train_labels,entest_zmp_fpiR,2);
        
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
             zmp_fpiR_model_1v1 = plda_twoCov_train22(numIter,train_zmp_fpiR_pls,train_labels,opt);
            
            
            
            %plda test scores
            [zmp_fpiR_plda_scores_1v1, lab_test_fpiR_1v1, gun_pairs_fpiR_1v1] = plda_twoCov_test(zmp_fpiR_model_1v1, entest_zmp_fpiR_pls,entest_labels,opt);
            
            
            
            % calibration & validation
            cllr_zmp_fpiR_1v1 = calibration_validation(zmp_fpiR_plda_scores_1v1,lab_test_fpiR_1v1,gun_pairs_fpiR_1v1,folder_to_store,1,'zmp fpiR 1v1');
            
            
            
            
        end
        
         % 1v3
        if do_1v3
            opt=2;
            
            % plda train
            zmp_fpiR_model_1v3 = plda_twoCov_train22(numIter,train_zmp_fpiR_pls,train_labels,opt);
            
            
            
            %plda test scores
            [zmp_fpiR_plda_scores_1v3, lab_test_fpiR_1v3, gun_pairs_fpiR_1v3] = plda_twoCov_test(zmp_fpiR_model_1v3, entest_zmp_fpiR_pls,entest_labels,opt);
            
            
            
            cllr_zmp_fpiR_1v3 = calibration_validation(zmp_fpiR_plda_scores_1v3,lab_test_fpiR_1v3,gun_pairs_fpiR_1v3,folder_to_store,1,'zmp fpiR 1v3');
            
            
        end
        
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            zmp_fpiR_model_1v9 = plda_twoCov_train22(numIter,train_zmp_fpiR_pls,train_labels,opt);
           
            
            
            %plda test scores
            [zmp_fpiR_plda_scores_1v9, lab_test_fpiR_1v9, gun_pairs_fpiR_1v9] = plda_twoCov_test(zmp_fpiR_model_1v9, entest_zmp_fpiR_pls,entest_labels,opt);
            
            
            
           cllr_zmp_fpiR_1v9 = calibration_validation(zmp_fpiR_plda_scores_1v9,lab_test_fpiR_1v9,gun_pairs_fpiR_1v9,folder_to_store,1,'zmp fpiR 1v9');
            
            
        end
        
       
        
    end
    if bfr|scf
%        
        train_zmp_bfrR=zmp_bfrR(idx,:);
               
        
        entest_zmp_bfrR=zmp_bfrR(idxET,:);
        
        
        
        % center data and estimate pca then lda and scale the data
        % Value=3, 20 PCA dim selected
        [train_zmp_bfrR_pls,entest_zmp_bfrR_pls] = pca_lda_scale(train_zmp_bfrR,train_labels,entest_zmp_bfrR,3);
        
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            zmp_bfrR_model_1v1 = plda_twoCov_train22(numIter,train_zmp_bfrR_pls,train_labels,opt);
            
            
            
            %plda test scores
            [zmp_bfrR_plda_scores_1v1, lab_test_bfrR_1v1, gun_pairs_bfrR_1v1] = plda_twoCov_test(zmp_bfrR_model_1v1, entest_zmp_bfrR_pls,entest_labels,opt);
            
            
            
            % calibration & validation
            cllr_zmp_bfrR_1v1 = calibration_validation(zmp_bfrR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'zmp bfrR 1v1');
           
            
            
            
        end
        
        % 1v3
        if do_1v3
            opt=2;
            
            % plda train
            zmp_bfrR_model_1v3 = plda_twoCov_train22(numIter,train_zmp_bfrR_pls,train_labels,opt);
            
            
            
            %plda test scores
            [zmp_bfrR_plda_scores_1v3, lab_test_bfrR_1v3, gun_pairs_bfrR_1v3] = plda_twoCov_test(zmp_bfrR_model_1v3, entest_zmp_bfrR_pls,entest_labels,opt);
            
            
            
            cllr_zmp_bfrR_1v3 = calibration_validation(zmp_bfrR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'zmp bfrR 1v3');
           
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            zmp_bfrR_model_1v9 = plda_twoCov_train22(numIter,train_zmp_bfrR_pls,train_labels,opt);
            
            
            
            %plda test scores
            [zmp_bfrR_plda_scores_1v9, lab_test_bfrR_1v9, gun_pairs_bfrR_1v9] = plda_twoCov_test(zmp_bfrR_model_1v9, entest_zmp_bfrR_pls,entest_labels,opt);
                       
            
            
            cllr_zmp_bfrR_1v9 = calibration_validation(zmp_bfrR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9,folder_to_store,1,'zmp bfrR 1v9');
            
            
        end
        
        
        
    end
    if wof

        train_zmp_wofR = zmp_wofR(idx,:);
        
        entest_zmp_wofR = zmp_wofR(idxET,:);
        
        
        
        % Value=1, 30 PCA dim selected
        [train_zmp_wofR_pls,entest_zmp_wofR_pls] = pca_lda_scale(train_zmp_wofR,train_labels,entest_zmp_wofR,1);
        
        
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            zmp_wofR_model_1v1 = plda_twoCov_train22(numIter,train_zmp_wofR_pls,train_labels,opt);
            
            
            
            %plda test scores
            [zmp_wofR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(zmp_wofR_model_1v1, entest_zmp_wofR_pls,entest_labels,opt);
            
            
            
            % calibration & validation
           cllr_zmp_wofR_1v1 = calibration_validation(zmp_wofR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'zmp wofR 1v1');
            
            
            
            
        end
        
         % 1v3
        if do_1v3
            opt=2;
            
            % plda train
            zmp_wofR_model_1v3 = plda_twoCov_train22(numIter,train_zmp_wofR_pls,train_labels,opt);
           
            
            
            %plda test scores
            [zmp_wofR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(zmp_wofR_model_1v3, entest_zmp_wofR_pls,entest_labels,opt);
            
            
            
            cllr_zmp_wofR_1v3 = calibration_validation(zmp_wofR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'zmp wofR 1v3');
           
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            zmp_wofR_model_1v9 = plda_twoCov_train22(numIter,train_zmp_wofR_pls,train_labels,opt);
           
            
            
            %plda test scores
           [zmp_wofR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(zmp_wofR_model_1v9, entest_zmp_wofR_pls,entest_labels,opt);
            
            
            
            cllr_zmp_wofR_1v9 = calibration_validation(zmp_wofR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'zmp wofR 1v9');
            
            
        end
        
       
    end
    
    if  cct    % feature level fusion
        % train data
        train_zmp_cctR=[zmp_bfrR(idx,:) zmp_fpiR(idx,:)];
        
        
        % enrol & test data
        entest_zmp_cctR=[zmp_bfrR(idxET,:) zmp_fpiR(idxET,:)];
        
        
        
        % center data and estimate pca then lda and scale the data
        % Value=1, 30 PCA dim selected
        [train_zmp_cctR_pls,entest_zmp_cctR_pls] = pca_lda_scale(train_zmp_cctR,train_labels,entest_zmp_cctR,1);
        
        
       
        
        % 1v1
        if do_1v1
            opt=1;
            % plda train
            zmp_cctR_model_1v1 = plda_twoCov_train22(numIter,train_zmp_cctR_pls,train_labels,opt);
            
            
            
            %plda test scores
            [zmp_cctR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(zmp_cctR_model_1v1, entest_zmp_cctR_pls,entest_labels,opt);
            
            
            
            % calibration & validation
            cllr_zmp_cctR_1v1 = calibration_validation(zmp_cctR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,1,'zmp cctR 1v1');
            
            
        end
        
         % 1v3
        if do_1v3
            opt=2;
            % plda train
            zmp_cctR_model_1v3 = plda_twoCov_train22(numIter,train_zmp_cctR_pls,train_labels,opt);
            
            
            
            %plda test scores
            [zmp_cctR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(zmp_cctR_model_1v3, entest_zmp_cctR_pls,entest_labels,opt);
            
            
            
            % calibration & validation
            cllr_zmp_cctR_1v3 = calibration_validation(zmp_cctR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,1,'zmp cctR 1v3');
            
            
            
        end
        
        % 1v9
        if do_1v9
            opt=3;
            % plda train
            zmp_cctR_model_1v9 = plda_twoCov_train22(numIter,train_zmp_cctR_pls,train_labels,opt);
            
            
            
            %plda test scores
            [zmp_cctR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(zmp_cctR_model_1v9, entest_zmp_cctR_pls,entest_labels,opt);
            
            
            
            
            % calibration & validation
             cllr_zmp_cctR_1v9 = calibration_validation(zmp_cctR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,1,'zmp cctR 1v9');
            
            
        end
        
        
       
        
    end
    
    if scf
        
        if do_1v1
            % for 1v1
            zmp_scfR_plda_scores_1v1= [zmp_bfrR_plda_scores_1v1 zmp_fpiR_plda_scores_1v1];
            
            
            
            cllr_zmp_scfR_1v1= calibration_validation(zmp_scfR_plda_scores_1v1,lab_test_bfrR_1v1,gun_pairs_bfrR_1v1,folder_to_store,1,'zmp scfR 1v1');
            
            
        end
        
        if do_1v3
            % for 1v3
            zmp_scfR_plda_scores_1v3=[zmp_bfrR_plda_scores_1v3 zmp_fpiR_plda_scores_1v3];
                        
            
            cllr_zmp_scfR_1v3= calibration_validation(zmp_scfR_plda_scores_1v3,lab_test_bfrR_1v3,gun_pairs_bfrR_1v3,folder_to_store,1,'zmp scfR 1v3');
           
            
        end
        
        if do_1v9
            % for 1v9
            zmp_scfR_plda_scores_1v9=[zmp_bfrR_plda_scores_1v9 zmp_fpiR_plda_scores_1v9];
            
                     
            
            cllr_zmp_scfR_1v9= calibration_validation(zmp_scfR_plda_scores_1v9,lab_test_bfrR_1v9,gun_pairs_bfrR_1v9,folder_to_store,1,'zmp scfR 1v9');
            
            
        end
        
        
        
    end
    
end

% To print all Cllr values into a csv file
if print2File
    
 wwf_val= [cllr_cem_wwfR_1v1 cllr_cim_wwfR_1v1 cllr_lmi_wwfR_1v1  cllr_cmi_wwfR_1v1 cllr_fma_wwfR_1v1 cllr_fmp_wwfR_1v1,...
     cllr_zma_wwfR_1v1 cllr_zmp_wwfR_1v1 cllr_cem_wwfR_1v3 cllr_cim_wwfR_1v3 cllr_lmi_wwfR_1v3  cllr_cmi_wwfR_1v3 cllr_fma_wwfR_1v3 cllr_fmp_wwfR_1v3,...
     cllr_zma_wwfR_1v3 cllr_zmp_wwfR_1v3 cllr_cem_wwfR_1v9 cllr_cim_wwfR_1v9 cllr_lmi_wwfR_1v9  cllr_cmi_wwfR_1v9 cllr_fma_wwfR_1v9 cllr_fmp_wwfR_1v9,...
     cllr_zma_wwfR_1v9 cllr_zmp_wwfR_1v9];
 
 wof_val= [cllr_cem_wofR_1v1 cllr_cim_wofR_1v1 cllr_lmi_wofR_1v1  cllr_cmi_wofR_1v1 nan nan,...
     cllr_zma_wofR_1v1 cllr_zmp_wofR_1v1 cllr_cem_wofR_1v3 cllr_cim_wofR_1v3 cllr_lmi_wofR_1v3  cllr_cmi_wofR_1v3 nan nan,...
     cllr_zma_wofR_1v3 cllr_zmp_wofR_1v3 cllr_cem_wofR_1v9 cllr_cim_wofR_1v9 cllr_lmi_wofR_1v9  cllr_cmi_wofR_1v9 nan nan,...
     cllr_zma_wofR_1v9 cllr_zmp_wofR_1v9];
 
 bfr_val= [cllr_cem_bfrR_1v1 cllr_cim_bfrR_1v1 cllr_lmi_bfrR_1v1  cllr_cmi_bfrR_1v1 nan nan,...
     cllr_zma_bfrR_1v1 cllr_zmp_bfrR_1v1 cllr_cem_bfrR_1v3 cllr_cim_bfrR_1v3 cllr_lmi_bfrR_1v3  cllr_cmi_bfrR_1v3 nan nan,...
     cllr_zma_bfrR_1v3 cllr_zmp_bfrR_1v3 cllr_cem_bfrR_1v9 cllr_cim_bfrR_1v9 cllr_lmi_bfrR_1v9  cllr_cmi_bfrR_1v9 nan nan,...
     cllr_zma_bfrR_1v9 cllr_zmp_bfrR_1v9];
 
 fpi_val= [cllr_cem_fpiR_1v1 cllr_cim_fpiR_1v1 cllr_lmi_fpiR_1v1  cllr_cmi_fpiR_1v1 nan nan,...
     cllr_zma_fpiR_1v1 cllr_zmp_fpiR_1v1 cllr_cem_fpiR_1v3 cllr_cim_fpiR_1v3 cllr_lmi_fpiR_1v3  cllr_cmi_fpiR_1v3 nan nan,...
     cllr_zma_fpiR_1v3 cllr_zmp_fpiR_1v3 cllr_cem_fpiR_1v9 cllr_cim_fpiR_1v9 cllr_lmi_fpiR_1v9  cllr_cmi_fpiR_1v9 nan nan,...
     cllr_zma_fpiR_1v9 cllr_zmp_fpiR_1v9];
 
 cct_val= [cllr_cem_cctR_1v1 cllr_cim_cctR_1v1 cllr_lmi_cctR_1v1  cllr_cmi_cctR_1v1 nan nan,...
     cllr_zma_cctR_1v1 cllr_zmp_cctR_1v1 cllr_cem_cctR_1v3 cllr_cim_cctR_1v3 cllr_lmi_cctR_1v3 cllr_cmi_cctR_1v3 nan nan,...
     cllr_zma_cctR_1v3 cllr_zmp_cctR_1v3 cllr_cem_cctR_1v9 cllr_cim_cctR_1v9 cllr_lmi_cctR_1v9  cllr_cmi_cctR_1v9 nan nan,...
     cllr_zma_cctR_1v9 cllr_zmp_cctR_1v9];
 
 scf_val= [cllr_cem_scfR_1v1 cllr_cim_scfR_1v1 cllr_lmi_scfR_1v1  cllr_cmi_scfR_1v1 nan nan,...
     cllr_zma_scfR_1v1 cllr_zmp_scfR_1v1 cllr_cem_scfR_1v3 cllr_cim_scfR_1v3 cllr_lmi_scfR_1v3  cllr_cmi_scfR_1v3 nan nan,...
     cllr_zma_scfR_1v3 cllr_zmp_scfR_1v3 cllr_cem_scfR_1v9 cllr_cim_scfR_1v9 cllr_lmi_scfR_1v9  cllr_cmi_scfR_1v9 nan nan,...
     cllr_zma_scfR_1v9 cllr_zmp_scfR_1v9];
 
 col_headers= [" ","wwf","wof","bfr","fpi","cct","scf"];
 
 row_headers=["cem_1v1","cim_1v1","lmi_1v1","cmi_1v1","fma_1v1","fmp_1v1","zma_1v1","zmp_1v1",...
     "cem_1v3","cim_1v3","lmi_1v3","cmi_1v3","fma_1v3","fmp_1v3","zma_1v3","zmp_1v3",...
     "cem_1v9","cim_1v9","lmi_1v9","cmi_1v9","fma_1v9","fmp_1v9","zma_1v9","zmp_1v9"];
 
 row_headers=row_headers';
 results=[wwf_val' wof_val' bfr_val' fpi_val' cct_val' scf_val'];
 cllr_file=strcat(folder_to_store,'results.txt');
 csvwrite(cllr_file,[]);
 fid = fopen(cllr_file,'w+');
 fprintf(fid, '%s\t%s\t%s\t%s\t%s\t%s\t%s\t\n', col_headers);
  for u=1:length(scf_val)
   fprintf(fid, '%s\t',row_headers(u));
   fprintf(fid, '%.3f\t%.3f\t%.3f\t%.3f\t%.3f\t%.3f\n',results(u,:));
 end
 fclose(fid);

end


%% 33 randomizations of guns
cllr_zmp_wwfR_randA_1v1=[]; cllr_zmp_wwfR_randA_1v9=[]; cllr_zmp_wwfR_randA_1v3=[];

if randomization
    if zmp
        if wwf
            uLab_idx = select_unique_sets(length(guns_to_consider),33,train_no_of_diff_guns);
            parfor (I_3=1:33,29)
                
                %Gun numbers used for training
                
                gun_nos_train=guns_to_consider(uLab_idx(I_3,:));
                gun_nos_enrol_test=setdiff(guns_to_consider,gun_nos_train);
                
                % training indices
                idx=[];
                for j=1:length(gun_nos_train)
                    idxn=(labs==gun_nos_train(j));
                    set=1:length(idxn);
                    eachGun=set(idxn==1);
                    idx=[idx;eachGun'];
                end
                
                % Training Labels
                train_labels=labs(idx);
                
                % Enrolment and test indices
                gun_nos_enrol_test=uLab_rand(train_no_of_diff_guns+1:end);
                
                idxET=[];
                for j=1:length(gun_nos_enrol_test)
                    idxE=(labs==gun_nos_enrol_test(j));
                    setE=1:length(idxE);
                    eachGunE=setE(idxE==1);
                    idxET=[idxET;eachGunE'];
                end
                
                % Enrolment and test Labels
                entest_labels=labs(idxET,:);
                
                train_zmp_wwfR=zmp_wwfR(idx,:);
                
                entest_zmp_wwfR=zmp_wwfR(idxET,:);
                
                
                
                % center data and estimate pca then lda and scale the data
                % Value=1, 30 PCA dim selected
                [train_zmp_wwfR_pls,entest_zmp_wwfR_pls] = pca_lda_scale(train_zmp_wwfR,train_labels,entest_zmp_wwfR,1);
                
                % 1v1
                if do_1v1
                    opt=1;
                    % plda train
                    zmp_wwfR_model_1v1 = plda_twoCov_train22(numIter,train_zmp_wwfR_pls,train_labels,opt);
                    
                    
                    %plda test scores
                    [zmp_wwfR_plda_scores_1v1, lab_test_1v1, gun_pairs_1v1] = plda_twoCov_test(zmp_wwfR_model_1v1, entest_zmp_wwfR_pls,entest_labels,opt);
                    
                    
                    % calibration & validation
                   cllr_zmp_wwfR_rand_1v1 = calibration_validation(zmp_wwfR_plda_scores_1v1,lab_test_1v1,gun_pairs_1v1,folder_to_store,0,'');
                    
                   cllr_zmp_wwfR_randA_1v1 =[cllr_zmp_wwfR_randA_1v1;cllr_zmp_wwfR_rand_1v1];
                end
                
                
                % 1v3
                if do_1v3
                    opt=2;
                    
                    % plda train
                    zmp_wwfR_model_1v3 = plda_twoCov_train22(numIter,train_zmp_wwfR_pls,train_labels,opt);
                    
                    
                    %plda test scores
                    [zmp_wwfR_plda_scores_1v3, lab_test_1v3, gun_pairs_1v3] = plda_twoCov_test(zmp_wwfR_model_1v3, entest_zmp_wwfR_pls,entest_labels,opt);
                    
                    
                    cllr_zmp_wwfR_rand_1v3  = calibration_validation(zmp_wwfR_plda_scores_1v3,lab_test_1v3,gun_pairs_1v3,folder_to_store,0,'');
                    
                    cllr_zmp_wwfR_randA_1v3 = [cllr_zmp_wwfR_randA_1v3;cllr_zmp_wwfR_rand_1v3];
                    
                    
                end
                
                if do_1v9
                    opt=3;
                    % plda train
                    zmp_wwfR_model_1v9 = plda_twoCov_train22(numIter,train_zmp_wwfR_pls,train_labels,opt);
                    
                    
                    %plda test scores
                    [zmp_wwfR_plda_scores_1v9, lab_test_1v9, gun_pairs_1v9] = plda_twoCov_test(zmp_wwfR_model_1v9, entest_zmp_wwfR_pls,entest_labels,opt);
                    
                    
                    cllr_zmp_wwfR_rand_1v9 = calibration_validation(zmp_wwfR_plda_scores_1v9,lab_test_1v9,gun_pairs_1v9,folder_to_store,0,'');
                
                    cllr_zmp_wwfR_randA_1v9=[cllr_zmp_wwfR_randA_1v9;cllr_zmp_wwfR_rand_1v9];
                end
                
                
                
            end
            
            cllr_v_cpd(cllr_zmp_wwfR_randA_1v1,cllr_zmp_wwfR_1v1,folder_to_store,'zmp wwfR 1v1 randomized');
            
            cllr_v_cpd(cllr_zmp_wwfR_randA_1v3,cllr_zmp_wwfR_1v3,folder_to_store,'zmp wwfR 1v3 randomized');
            
            cllr_v_cpd(cllr_zmp_wwfR_randA_1v9,cllr_zmp_wwfR_1v9,folder_to_store,'zmp wwfR 1v9 randomized');
            
            cllr_rand=[cllr_zmp_wwfR_randA_1v1,cllr_zmp_wwfR_randA_1v3,cllr_zmp_wwfR_randA_1v9];
            
            
            fileName='cllr_rand.mat';
            full_file=fullfile(folder_to_store,fileName);
            
            save(full_file,'cllr_rand');
  
        end
    end
end